//
//  TWChannelsCollectionViewController.m
//  Xarth
//
//  Created by Auston Stewart on 10/22/13.
//  Copyright (c) 2013 Justin.tv, Inc. All rights reserved.
//

#import "TWLiveChannelsCollectionViewController.h"
#import "TWStreamsCollectionViewCell.h"
#import "TWStreamTableViewCell.h"
#import "TWEnlargedStreamTableViewCell.h"
#import "TWAppDelegate.h"
#import "TWNoContentOverlayView.h"
#import "NSString+Twitch.h"
#import "TWAnalyticsController.h"

@interface TWLiveChannelsCollectionViewController () <TWCacheDelegate, UIPopoverControllerDelegate>
{
	NSString *_gameTitle;
	NSString *_requestIdentifier;
	NSString *_enlargedImageSizeKey;
	NSString *_imageSizeKey;
	TWCache *_cache;
	
	UIBarButtonItem *_optionsBarButtonItem;
	UIPopoverController *_optionsPopoverController;
}
@end

@implementation TWLiveChannelsCollectionViewController

#pragma mark Overridden APICollectionView Methods

+ (UICollectionViewFlowLayout *)flowLayout
{
	UICollectionViewFlowLayout *flowLayout = [[UICollectionViewFlowLayout alloc] init];
	flowLayout.scrollDirection = UICollectionViewScrollDirectionVertical;
	flowLayout.itemSize = [TWStreamsCollectionViewCell cellSize];
	flowLayout.minimumInteritemSpacing = TWGridContentInset;
	flowLayout.minimumLineSpacing = TWGridContentInset;
	flowLayout.sectionInset = UIEdgeInsetsMake(TWGridContentInset, TWGridContentInset, TWGridContentInset, TWGridContentInset);
	
	return flowLayout;
}

- (void)commonInitialization
{
	[super commonInitialization];
	
	_displaysBroadcastTitles = YES;
	_cache = [[TWCache alloc] initWithIdentifier:@"TWStreamsListCache"];
	_cache.delegate = self;
	self.title = TKLocalizedString(@"Channels", @"Channels");
	self.preferredStreamPresentationMode = TKStreamPresentationModeUseTwitchAppIfInstalled;
	self.referralIdentity = TWAnalyticsReferrerChannels;
	self.comparator = ^(id one, id two) {
		TKStream *streamOne = (TKStream *)one;
		TKStream *streamTwo = (TKStream *)two;
		
		if (streamOne.viewers > streamTwo.viewers)
			return NSOrderedAscending;
		if (streamOne.viewers < streamTwo.viewers)
			return NSOrderedDescending;
		return NSOrderedSame;
	};
	_enlargedImageSizeKey = [UIScreen mainScreen].scale > 1.f ? @"large" : @"medium";
	_imageSizeKey = [UIScreen mainScreen].scale > 1.f ? @"medium" : @"small";
	
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(requestDidFinish:) name:TKStreamListRequestDidFinishNotification object:nil];
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(requestDidFail:) name:TKStreamListRequestDidFailNotification object:nil];
}

- (void)viewDidLoad
{
	[super viewDidLoad];
		
	if ([UIDevice currentDevice].isPad) {
		
		[self.collectionView registerClass:[TWStreamsCollectionViewCell class] forCellWithReuseIdentifier:TWStreamsCollectionViewCellReuseIdentifier];
	}
	else {
		
		// Adjust content inset to create a proper top margin for grid-like cells
		UIEdgeInsets contentInset = self.tableView.contentInset;
		if (self.tableViewStyle == TWAPITableViewStyleMixed || self.tableViewStyle == TWAPITableViewStyleGrid)
			contentInset.top += TWGridContentInset;
		self.tableView.contentInset = contentInset;
		
		self.tableView.separatorStyle = UITableViewCellSeparatorStyleNone;
		[self.tableView registerClass:[TWStreamTableViewCell class] forCellReuseIdentifier:[TWStreamTableViewCell reuseIdentifier]];
		[self.tableView registerClass:[TWEnlargedStreamTableViewCell class] forCellReuseIdentifier:[TWEnlargedStreamTableViewCell reuseIdentifier]];
	}
}

- (void)viewWillAppear:(BOOL)animated
{
	[super viewWillAppear:animated];
	
	// On iPhone, set the proper orientation
	if ([UIDevice currentDevice].isPhone) {
		
		[self updateForOrientation:self.interfaceOrientation];
	}
}

- (id)init
{
	if ([UIDevice currentDevice].isPhone) {
		
		TWAPITableViewStyle defaultTableViewStyle = [[NSUserDefaults standardUserDefaults] integerForKey:TWLiveChannelListPresentationDefaultsKey];
		return [super initWithTableViewStyle:defaultTableViewStyle];
	}
	else
		return [super initWithCollectionViewLayout:[[self class] flowLayout]];
}

- (id)initWithGameTitle:(NSString *)gameTitle
{
	if ((self = [self init])) {
		
		_gameTitle = [gameTitle copy];
		self.title = _gameTitle;
	}
	
	return self;
}

- (void) popoverControllerDidDismissPopover:(id) popoverControllerToDismiss {
	
	_optionsPopoverController = nil;
}

- (void)updateViewForContentChange
{
	// Show the no content overlay if we have no items following the update
	if (!self.items.count) {
		
		self.noContentOverlayView.backgroundColor = [UIColor whiteColor];
		self.noContentOverlayView.icon = [[UIImage imageFromFrameworkBundleNamed:@"glyph_graphic_blankslate_channels"] imageWithRenderingMode:UIImageRenderingModeAlwaysTemplate];
		self.noContentOverlayView.title = TKLocalizedString(@"Aww, you missed them",@"No stream results overlay title");
		self.noContentOverlayView.message = _gameTitle.length ? [NSString stringWithFormat:TKLocalizedString(@"There are currently no broadcasts\nof %@.\n\nDon't worry, there are plenty more\ngames available for you to watch!", @"No game stream results overlay message"),_gameTitle] : TKLocalizedString(@"There are currently no live broadcasts.", @"No stream results overlay message");
		self.noContentOverlayView.hidden = NO;
	}
	else self.noContentOverlayView.hidden = YES;
}

- (BOOL)shouldProcessResponseWithTags:(NSDictionary *)requestTags
{
	return [[requestTags objectForKey:@"identifier"] isEqualToString:_requestIdentifier];
}

- (void)requestItemsFromOffset:(NSUInteger)offset limit:(NSUInteger)limit
{
	TWHTTPRequest *streamsRequest = [TWHTTPRequest krakenStreamListRequest];
	if (!self.includeUnplayableContent) {
		
		[streamsRequest addParameter:@"hls" withValue:@"true"];
		[UIDevice setRequestParametersForVideoCapabilities:streamsRequest];
	}
	if (self.gameTitle) {
		[streamsRequest addParameter:@"game" withValue:self.gameTitle];
	}
	[streamsRequest addParameter:@"offset" withValue:[NSString stringWithFormat:@"%d",offset]];
	[streamsRequest addParameter:@"limit" withValue:[NSString stringWithFormat:@"%d",limit]];
	
	_requestIdentifier = [NSString tw_UUIDString];
	[streamsRequest setTag:@"identifier" withValue:_requestIdentifier];
	streamsRequest.requestTimeoutInterval = 5.;
	
	streamsRequest.responseChecksAvailableCache = NO;
	[self.operationQueue addOperation:streamsRequest];
}

- (void)setTableViewStyle:(TWAPITableViewStyle)tableViewStyle
{
	if (self.tableViewStyle != tableViewStyle) {
		
		TWAPITableViewStyle oldTableViewStyle = self.tableViewStyle;
		[super setTableViewStyle:tableViewStyle];
		
		UIEdgeInsets contentInset = self.tableView.contentInset;
		
		if ((oldTableViewStyle == TWAPITableViewStyleGrid || oldTableViewStyle == TWAPITableViewStyleMixed) && self.tableViewStyle == TWAPITableViewStyleList) {
			contentInset.top -= TWGridContentInset;
		}
		else if (oldTableViewStyle == TWAPITableViewStyleList && (self.tableViewStyle == TWAPITableViewStyleGrid || self.tableViewStyle == TWAPITableViewStyleMixed)) {
			contentInset.top += TWGridContentInset;
		}
		
		self.tableView.contentInset = contentInset;
	}
}

- (void)setDisplaysBroadcastTitles:(BOOL)displaysBroadcastTitles
{
	if (self.displaysBroadcastTitles != displaysBroadcastTitles) {
		
		_displaysBroadcastTitles = displaysBroadcastTitles;
		[self.tableView reloadData];
		[self.collectionView reloadData];
	}
}

#pragma mark UICollectionViewDatasource Methods

- (UICollectionViewCell *)collectionView:(UICollectionView *)collectionView cellForItemAtIndexPath:(NSIndexPath *)indexPath
{
	TKStream *stream = self.items[indexPath.row];
	TWStreamsCollectionViewCell *cell = [collectionView dequeueReusableCellWithReuseIdentifier:TWStreamsCollectionViewCellReuseIdentifier forIndexPath:indexPath];
	
	if (_displaysBroadcastTitles) cell.broadcastTitle = stream.channel.status;
	cell.channelDisplayName = stream.channel.displayName;
	cell.viewerCount = stream.viewers;
	if (stream.game && !_gameTitle) cell.gameName = [stream.game tw_stringByAbridgingGameTitle];
	[cell loadImageAtURL:stream.preview[_enlargedImageSizeKey] fromCache:_cache persistently:NO];
	
	return cell;
}

#pragma mark UICollectionViewDelegate Methods

- (void)collectionView:(UICollectionView *)collectionView didSelectItemAtIndexPath:(NSIndexPath *)indexPath
{
	TKStream *stream = self.items[indexPath.row];
	[[TwitchKit sharedPlugin] presentStreamForChannelNamed:stream.channel.name mode:self.preferredStreamPresentationMode];
	[collectionView deselectItemAtIndexPath:indexPath animated:YES];
	
	NSMutableDictionary *analyticProperties = [NSMutableDictionary dictionaryWithCapacity:2];
	if (_gameTitle.length) analyticProperties[TWAnalyticsPropertyGame] = _gameTitle;
	analyticProperties[TWAnalyticsPropertyChannel] = stream.channel.name;
	[[TWAnalyticsController analyticsController] trackEvent:TWAnalyticsEventStreamPlay withDetails:analyticProperties];
}

#pragma mark UITableViewDatasource Methods

- (CGFloat)tableView:(UITableView *)tableView heightForRowAtIndexPath:(NSIndexPath *)indexPath
{
	if (self.tableViewStyle == TWAPITableViewStyleGrid ||
		(self.tableViewStyle == TWAPITableViewStyleMixed && self.items.count < 10) ||
		(self.tableViewStyle == TWAPITableViewStyleMixed && indexPath.row < 5)) {
		
		// Create a margin between items
		return [TWEnlargedStreamTableViewCell preferredHeight] + 10.f;
	}
	
	return [TWStreamTableViewCell preferredHeight];
}

- (CGFloat)tableView:(UITableView *)tableView estimatedHeightForRowAtIndexPath:(NSIndexPath *)indexPath
{
	return [self tableView:tableView heightForRowAtIndexPath:indexPath];
}

- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath
{
	TKStream *stream = self.items[indexPath.row];
	TWStreamTableViewCell *cell = nil;
	BOOL isEnlarged = NO;
	
	if (self.tableViewStyle == TWAPITableViewStyleGrid ||
		(self.tableViewStyle == TWAPITableViewStyleMixed && self.items.count < 10) ||
		(self.tableViewStyle == TWAPITableViewStyleMixed && indexPath.row < 5)) {
		
		cell = [tableView dequeueReusableCellWithIdentifier:[TWEnlargedStreamTableViewCell reuseIdentifier] forIndexPath:indexPath];
		cell.contentVerticalAlignment = TWTableViewCellContentVerticalAlignmentTop;
		isEnlarged = YES;
		cell.showsBottomSeparator = (self.tableViewStyle == TWAPITableViewStyleMixed && (indexPath.row == 4));
	}
	else {
		
		cell = [tableView dequeueReusableCellWithIdentifier:[TWStreamTableViewCell reuseIdentifier] forIndexPath:indexPath];
		isEnlarged = NO;
		cell.showsBottomSeparator = YES;
	}
	
	if (_displaysBroadcastTitles) cell.broadcastTitle = stream.channel.status;
	cell.channelDisplayName = stream.channel.displayName;
	cell.viewerCount = stream.viewers;
	if (stream.game && !_gameTitle) cell.gameName = [stream.game tw_stringByAbridgingGameTitle];
	[cell loadImageAtURL:stream.preview[isEnlarged ? _enlargedImageSizeKey : _imageSizeKey] fromCache:_cache persistently:NO];
	return cell;
}

#pragma mark UITableViewDelegate Methods

- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath
{
	TKStream *stream = self.items[indexPath.row];
	[[TwitchKit sharedPlugin] presentStreamForChannelNamed:stream.channel.name mode:self.preferredStreamPresentationMode];
	[tableView deselectRowAtIndexPath:indexPath animated:YES];
	
	NSMutableDictionary *analyticProperties = [NSMutableDictionary dictionaryWithCapacity:2];
	if (_gameTitle.length) analyticProperties[TWAnalyticsPropertyGame] = _gameTitle;
	analyticProperties[TWAnalyticsPropertyChannel] = stream.channel.name;
	[[TWAnalyticsController analyticsController] trackEvent:TWAnalyticsEventStreamPlay withDetails:analyticProperties];
}

#pragma mark TWCacheDelegate Methods

- (NSDate *) evictionDateForObject:(id) object fromKey:(id) key {
	return [[NSDate date] dateByAddingTimeInterval:300.]; // Refresh live feed images every 5 minutes
}

#pragma mark UICollectionViewFlowLayoutDelegate Methods
	
- (CGSize)collectionView:(UICollectionView *)collectionView layout:(UICollectionViewLayout  *)collectionViewLayout sizeForItemAtIndexPath:(NSIndexPath *)indexPath
{
	// Adjust cell size for orientation
	if (UIDeviceOrientationIsLandscape([[UIApplication sharedApplication] statusBarOrientation])) {
		return [TWStreamsCollectionViewCell landscapeCellSize];
	}
	return [TWStreamsCollectionViewCell portraitCellSize];
}

#undef TWLiveChannelsCollectionViewCellPadding

- (void)updateForOrientation:(UIInterfaceOrientation)interfaceOrientation
{
	if (UIInterfaceOrientationIsPortrait(interfaceOrientation) || [UIDevice currentDevice].isPad) {
		
		// Grid view in portrait
		self.tableViewStyle = TWAPITableViewStyleGrid;
		
		// Normal sized logo
		self.navigationItem.titleView = [[UIImageView alloc] initWithImage:[UIImage imageFromFrameworkBundleNamed:@"logo_navbar_portrait"]];
	}
	else {
		
		// List style in landscape
		self.tableViewStyle = TWAPITableViewStyleList;
		
		// Landscape logo
		self.navigationItem.titleView = [[UIImageView alloc] initWithImage:[UIImage imageFromFrameworkBundleNamed:@"logo_navbar_landscape"]];
	}
}

- (void)willAnimateRotationToInterfaceOrientation:(UIInterfaceOrientation)toInterfaceOrientation duration:(NSTimeInterval)duration
{
	[super willAnimateRotationToInterfaceOrientation:toInterfaceOrientation duration:duration];
	
	if ([UIDevice currentDevice].isPad) {
		
		// Invalidate layout to resize grid items
		[self.collectionView.collectionViewLayout invalidateLayout];
	}
	else {
		
		// On iPhone switch tableview modes
		[self updateForOrientation:toInterfaceOrientation];
	}
}
	
@end
